import math
#Some important physical constants in mks
c = 299792458.0
mu = 4*(math.pi)*(1e-07)
fine_struct = 7.297352570e-3

#Window and photocathode material types: d is thickness of window, b & c are Sellmeier coefficients
#file is where the transmission data is, name is the name of the material

QE58         = {"name"      : "QE58",
		"file"      : "photocathode/R3809U-58.dat"}
		
testcath     = {"name"      : "Test Cathode",
		"file"      : "photocathode/testcath.dat"}

fused_silica = {"name"      : "Fused Silica 2.5mm",
		"d"         :  0.0025,
                "b"         : (0.68374049400, 0.42032361300, 0.58502748000),
                "c"         : (0.00460352869, 0.01339688560, 64.49327320000),
		"file"      : "window/transSilica.dat"}
		
MgF2         = {"name"      : "MgF2",
		"d"         :  0.0025,
                "b"         : (0.48755108, 0.39875031, 2.31203530),
                "c"         : (0.00135737865, .00823767167, 56.5107755),
		"file"      : "window/transMgF2.dat"}
		
fused_silica2 = {"name"     : "Fused Silica 2.0mm",
		"d"         :  0.002,
                "b"         : (0.68374049400, 0.42032361300, 0.58502748000),
                "c"         : (0.00460352869, 0.01339688560, 64.49327320000),
		"file"      : "window/transSilica.dat"}
		
fused_silica5 = {"name"     : "Fused Silica 5.0mm",
		"d"         :  0.005,
                "b"         : (0.68374049400, 0.42032361300, 0.58502748000),
                "c"         : (0.00460352869, 0.01339688560, 64.49327320000),
		"file"      : "window/transSilica.dat"}
		
fused_silica8 = {"name"     : "Fused Silica 8.0mm",
		"d"         :  0.008,
                "b"         : (0.68374049400, 0.42032361300, 0.58502748000),
                "c"         : (0.00460352869, 0.01339688560, 64.49327320000),
		"file"      : "window/transSilica.dat"}
		
fused_silica11 = {"name"    : "Fused Silica 11.0mm",
		"d"         :  0.011,
                "b"         : (0.68374049400, 0.42032361300, 0.58502748000),
                "c"         : (0.00460352869, 0.01339688560, 64.49327320000),
		"file"      : "window/transSilica.dat"}	
		
window_list = [fused_silica, fused_silica2, fused_silica5, fused_silica8, fused_silica11, MgF2]
photocathode_list = [QE58]


from datread import *

#Uses Sellmeier formula to calculate n for given wavelength		
def n(lam, material = fused_silica):
	return math.sqrt((material["b"][0]*(lam**2))/((lam**2)-material["c"][0]) + \
	                 (material["b"][1]*(lam**2))/((lam**2)-material["c"][1]) + \
		         (material["b"][2]*(lam**2))/((lam**2)-material["c"][2]) + 1)

#Class for incoming particle:
#self.b gives the particle's velocity vector, self.q gives its charge in e's, and
#this completely characterizes the particle for our purposes
#We rotate the particle so that the y-component=0 for easier computation			 
class Particle:
	def __init__(self, beta_x, beta_y, beta_z, q):
		beta_x = math.sqrt((beta_x**2)+(beta_y**2))
		beta_y = 0
		self.b = (float(beta_x), float(beta_y), float(beta_z))
		self.q = float(q)
		self.s = math.sqrt((beta_x**2)+(beta_z**2))
			 
#Calculates magnitude of beta, ie, incident particle speed
def speed(p):
	return math.sqrt(p.b[0]**2 + p.b[1]**2 + p.b[2]**2)

#Takes 3-vector of incident charged particle, returns cherenkov cone half-angle			 	
def angle(p, lam):
	if p.s*n(lam) < 1:	return 0
	else: return math.acos(1/(n(lam)*p.s))
	
def d2N(p, lam, material):
	ncons = n(250, material)
	return 2*(math.pi)*(p.q**2)*(fine_struct)*(1e9)*(1-((p.s*ncons)**(-2)))/(ncons*(lam**2))


#Determines the range of wavelengths which fulfill the relation: (beta)*(n(lamda)) >= 1
#Only wavelengths in this range should be used in the Frank-Tamm formula
#There's a very small range of values, speed around .6097 where the range isn't [160-Inf] or [0,0]
#Basically, if it's going fast enough for Cherenkov radiation, any wavelength will do...
#Remove this if it causes problems.
#In nanometers
def bound(p, material = fused_silica, photocathode = QE58):
	n0 = int((listgen(material)[0][0])*10)
	nmax = 5000
	a = 1
	lim = []
	lammax = listgen(photocathode)[-1][0]
	for elem in range(n0, nmax+1):
		elem = float(elem)/10.0
		if n(elem)*p.s >= 1.0:
			if a:
				li = []
				li.append(elem)
				a = 0
		else:
			if not a:
				li.append(elem)
				lim.append(li)
				a = 1
	if lim == []:
		if a: lim.append([0,0])
		if not a: lim.append([listgen(material)[0][0], lammax])
	return lim
